import { RealtimeConnection } from "./connection";
export declare enum AudioFormat {
    PCM_8000 = "pcm_8000",
    PCM_16000 = "pcm_16000",
    PCM_22050 = "pcm_22050",
    PCM_24000 = "pcm_24000",
    PCM_44100 = "pcm_44100",
    PCM_48000 = "pcm_48000",
    ULAW_8000 = "ulaw_8000"
}
export declare enum CommitStrategy {
    MANUAL = "manual",
    VAD = "vad"
}
interface BaseOptions {
    /**
     * Token to use for the WebSocket connection. Obtained from the ElevenLabs API.
     */
    token: string;
    /**
     * Strategy for committing transcriptions.
     * @default CommitStrategy.MANUAL
     */
    commitStrategy?: CommitStrategy;
    /**
     * Silence threshold in seconds for VAD (Voice Activity Detection).
     * Must be a positive number between 0.3 and 3.0
     */
    vadSilenceThresholdSecs?: number;
    /**
     * Threshold for voice activity detection.
     * Must be between 0.1 and 0.9.
     */
    vadThreshold?: number;
    /**
     * Minimum speech duration in milliseconds.
     * Must be a positive integer between 50 and 2000.
     */
    minSpeechDurationMs?: number;
    /**
     * Minimum silence duration in milliseconds.
     * Must be a positive integer between 50 and 2000.
     */
    minSilenceDurationMs?: number;
    /**
     * Model ID to use for transcription.
     * Must be a valid model ID.
     */
    modelId: string;
    /**
     * An ISO-639-1 or ISO-639-3 language_code corresponding to the language of the audio file.
     * Can sometimes improve transcription performance if known beforehand.
     */
    languageCode?: string;
    /**
     * Base URI to use for the WebSocket connection.
     * If not provided, the default URI will be used.
     */
    baseUri?: string;
}
export interface AudioOptions extends BaseOptions {
    audioFormat: AudioFormat;
    sampleRate: number;
    microphone?: never;
}
/**
 * Options for automatic microphone streaming in the browser.
 */
export interface MicrophoneOptions extends BaseOptions {
    microphone?: {
        deviceId?: string;
        echoCancellation?: boolean;
        noiseSuppression?: boolean;
        autoGainControl?: boolean;
        channelCount?: number;
    };
    audioFormat?: never;
    sampleRate?: never;
}
/**
 * Real-time speech-to-text transcription client for browser environments.
 * Supports microphone streaming and manual audio chunk transmission.
 */
export declare class ScribeRealtime {
    private static readonly DEFAULT_BASE_URI;
    private static getWebSocketUri;
    private static buildWebSocketUri;
    /**
     * Establishes a WebSocket connection for real-time speech-to-text transcription.
     *
     * @param options - Configuration options for the connection
     * @returns A RealtimeConnection instance
     *
     * @example
     * ```typescript
     * // Manual audio streaming
     * const connection = Scribe.connect({
     *     token: "...",
     *     modelId: "scribe_v2_realtime",
     *     audioFormat: AudioFormat.PCM_16000,
     *     sampleRate: 16000,
     * });
     *
     * // Automatic microphone streaming
     * const connection = Scribe.connect({
     *     token: "...",
     *     modelId: "scribe_v2_realtime",
     *     microphone: {
     *         echoCancellation: true,
     *         noiseSuppression: true
     *     }
     * });
     * ```
     */
    static connect(options: AudioOptions | MicrophoneOptions): RealtimeConnection;
    private static streamFromMicrophone;
}
export {};
